#include <string.h>
#include <errno.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>

#ifndef UNITTEST
#include <sys/neutrino.h>
#include "input/mtouch_driver.h"
#include <input/mtouch_log.h>
#endif

#include "touch.h"
#include "read_cm_cp_ref.h"
#include "save_cm_cp.h"

int
cypress_config_clear_error_pin(cypress_dev_t* cypress)
{
    uint8_t *reply = NULL;

    if ((cypress_config_cmd(cypress, CYPRESS_CONFIG_CLEAR_ERRORS, NULL, 0, &reply, CYPRESS_CONFIG_CLEAR_ERRORS_REPLY_LEN)) == -1) {
        mtouch_error (cypress->log_name, "Clear Errors failed");
        return -1;
    }

    if (!*reply)
        mtouch_info (cypress->log_name, "Clear Errors successful");
    else
        mtouch_error (cypress->log_name, "Failed to clear error pin");

    if (*reply)
        *reply = -1;

    return *reply;
}

int cypress_initialize_haptic_cmd(cypress_dev_t* cd, uint8_t *buff, uint16_t len)
{
    uint16_t offset;
    uint8_t haptic_mode;
    uint8_t *reply = NULL;
    int remaining_length = len, new_len;
    uint8_t *cmd_buf;
    int data_offset = 0;

    if(buff == NULL)
        return -EINVAL;

    offset = buff[0] | (buff[1]<<8);
    haptic_mode = buff[3] & 3;

    if (haptic_mode == 0) {
        mtouch_error(cd->log_name, "Invalid Haptic Mode:0x%02X\n", haptic_mode);
        return -1;
    }

    mtouch_debug(cd->log_name, "Offset:0x%04X, Data length:0x%02X,Haptic Mode:0x%01X\n",
        offset, buff[2], haptic_mode);

    do {
        if ((remaining_length + ( CONFIGURATION_COMMAND_DATA + 1 )) > CYPRESS_MAX_REGLEN ) {
            new_len = CYPRESS_MAX_REGLEN - ( CONFIGURATION_COMMAND_DATA + 1);
        }
        else {
            new_len = remaining_length;
        }

        cmd_buf =(uint8_t *) calloc (new_len, sizeof (uint8_t));

        if (NULL == cmd_buf) {
            mtouch_error (cd->log_name, "Failed to allocate memory for haptic data");
            return -1;
        }

        memcpy (cmd_buf, (buff + data_offset), new_len);

        if ((cypress_config_cmd(cd, CYPRESS_CONFIG_SET_HAPTIC_DATA, cmd_buf, new_len, &reply, CYPRESS_CONFIG_SET_HAPTIC_DATA_REPLY_LEN)) == -1) {
            mtouch_error (cd->log_name, "Set haptic data command failed");
            free(cmd_buf);
            return -1;
        }

        remaining_length-= new_len;
        data_offset += new_len;
        free(cmd_buf);

        if (!*reply) {
            mtouch_info (cd->log_name, "Set haptic data successful %d", *reply);
        }
        else {
            mtouch_error (cd->log_name, "Failed to set haptic data %d", *reply);
            *reply = -1;
            return *reply;
        }

   } while (remaining_length);

    if (!*reply)
        mtouch_info (cd->log_name, "Set haptic data successful");
    else
        mtouch_error (cd->log_name, "Failed to set haptic data reply: %d", *reply);

    if (*reply)
        *reply = -1;

    return *reply;
}

int cypress_set_haptic_data(cypress_dev_t* cd, uint8_t *haptic_data, uint16_t len)
{
    uint8_t mode = UNKNOWN;
#ifndef BOSCH_RTC_2487095_HID_CYPRESS_DATASHEET_ALIGNMENT_CHANGES
    uint8_t cur_drv_state = 0;
#endif
    int rc, ret;
    bool lock = 0;
    int haptic_status = 0;
    cypress_msg_u msg;
    cd->repeatCount = 0;
    struct itimerspec remaining_time;
    struct itimerspec pause_timer = {0};
    uint64_t timeout = 200 * 1000000;
    struct sigevent event;

    event.sigev_notify = SIGEV_UNBLOCK;

    rc = pthread_mutex_lock(&cd->tp_event_mutex);
    if (rc) {
         mtouch_error(cd->log_name, "Can't lock tp_event_mutex. Err: %d\n", rc);
         return rc;
    }
    lock = 1;

    /*Get the remaining time */
    timer_gettime(cd->heartbeat_timerid, &remaining_time);

    /* pause the heartbeat timer */
    if (EOK != timer_settime(cd->heartbeat_timerid, 0, &pause_timer, NULL)) {
        mtouch_error(cd->log_name, "%s failed to set heartbeat timer", __FUNCTION__);
        haptic_status = -1;
        goto fail1;
    }

    rc = pthread_mutex_unlock(&cd->tp_event_mutex);
    if (rc) {
         mtouch_error(cd->log_name, "Can't unlock tp_event_mutex. Err: %d\n", rc);
         haptic_status = -1;
         goto fail2;
    }
    lock = 0;

cat_mode_re_request:
    if ((mode = cypress_get_mode(cd)) != CONFIGURATION_AND_TEST_MODE) {
        if (cd->intr_coid != -1) {
#ifndef BOSCH_RTC_2487095_HID_CYPRESS_DATASHEET_ALIGNMENT_CHANGES
            cur_drv_state = cd->driver_state;
#endif
            msg.cmd = CYPRESS_MODE_CHG_REQUEST;
            msg.mode_chg.mode = CONFIGURATION_AND_TEST_MODE;
            rc = TimerTimeout(CLOCK_MONOTONIC, _NTO_TIMEOUT_SEND, &event, &timeout, NULL );
            if (-1 == rc) {
                mtouch_error(cd->log_name,"Timer_timeout failed for CAT mode change request, %s", strerror(errno));
            }
            rc = MsgSend(cd->intr_coid, &msg, sizeof(msg), NULL, 0);
            if(rc == CAT_MODE_CHANGE_RE_REQUEST){
                if (cd->repeatCount < REPEAT_COUNT_INFINEON)
		     goto cat_mode_re_request;
            }
            else if(rc != EOK) {
                mtouch_error(cd->log_name, "Error on request set mode to CAT rc=%d ,errno=%d\n", rc,errno);
                haptic_status = rc;
                goto fail3;
            }
            else
                mtouch_info(cd->log_name, "request to set mode to CAT success rc=%d\n", rc);
        } else {
            mtouch_info(cd->log_name, "Invalid interrupt thread connection id\n");
        }
    }

    rc = pthread_mutex_lock(&cd->tp_event_mutex);
    if (rc) {
         mtouch_error(cd->log_name, "Can't lock tp_event_mutex. Err: %d\n", rc);
         haptic_status = rc;
         goto fail3;
    }
    lock = 1;

    mtouch_info(cd->log_name, "Set controller to CAT mode success");

    rc = cypress_initialize_haptic_cmd(cd, haptic_data, len);
    if (rc < 0) {
        mtouch_error(cd->log_name, "Set Haptic Config Data command Failed.rc=%d\n",rc);
        haptic_status = rc;
    }
#ifdef BOSCH_RTC_2487095_HID_CYPRESS_DATASHEET_ALIGNMENT_CHANGES
    cd->haptic_write_status = true;
#endif

fail3:
#ifdef BOSCH_RTC_2487095_HID_CYPRESS_DATASHEET_ALIGNMENT_CHANGES
    if (lock) {
        ret = pthread_mutex_unlock(&cd->tp_event_mutex);
        if (ret) {
             mtouch_error(cd->log_name, "Can't unlock tp_event_mutex. Err: %d\n", ret);
        }
        lock = 0;
    }

    if (cd->intr_coid != -1)
    {
        msg.cmd = CYPRESS_MODE_CHG_REQUEST;
        msg.mode_chg.mode = OPERATING_MODE;
        rc = TimerTimeout(CLOCK_MONOTONIC, _NTO_TIMEOUT_SEND, &event, &timeout, NULL);
        if (-1 == rc) {
            mtouch_error(cd->log_name,"timer_timeout Failed for CAT to Operating mode change request, %s", strerror(errno));
            haptic_status = rc;
            goto fail2;
        }
        rc = MsgSend(cd->intr_coid, &msg, sizeof(msg), NULL, 0);
        if(rc != EOK)
        {
            mtouch_error(cd->log_name, "Error on request change mode back to OPERATING mode rc=%d, errno=%d\n", rc,errno);
            haptic_status = rc;
            goto fail2;
        }
        else{
            mtouch_info(cd->log_name, "Request to set mode to OPERATING success rc=%d\n", rc);
        }
    }
    else {
        mtouch_error(cd->log_name,"Interrupt thread connection id is invalid\n");
    }
#else
    ret = cypress_set_mode(cd, mode);
    if (ret < 0) {
        mtouch_error(cd->log_name, "Error on request set mode back to original state r=%d\n",ret);
        goto fail2;
    }
    cd->driver_state = cur_drv_state;
#endif

fail2:
    /* Resume the heartbeat timer */
    if (EOK != timer_settime(cd->heartbeat_timerid, 0, &remaining_time, NULL)) {
        mtouch_error (cd->log_name, "%s failed to set heartbeat timer", __FUNCTION__);
        error_memory ("Cypress_Touch: Failed to set the heartbeat timer to %ld sec %ld nsec", remaining_time.it_value.tv_sec, remaining_time.it_value.tv_nsec);
    }

fail1:
    if (lock) {
        ret = pthread_mutex_unlock(&cd->tp_event_mutex);
        if (ret) {
             mtouch_error(cd->log_name, "Can't unlock tp_event_mutex. Err: %d\n", ret);
        }
        lock = 0;
    }

    if (cd->haptic_flag)
        cd->haptic_flag = false;

    if (haptic_status)
        return haptic_status;
    else
        return rc;

}

/* Used for Self Tests and Retrieve Data Structure */
int
cypress_get_indexed_data(cypress_dev_t* cypress, uint8_t cmd, uint8_t data_id, uint8_t *data, int data_len)
{
    uint8_t buf[5] = {0};
    int bytes_remaining = data_len, bytes_received = 0, total_bytes_received = 0, bytes_to_read = 0;
    int status_offset = -1, id_offset = -1, data_offset = 0, bytes_received_offset = 0;
    uint8_t *reply = NULL;

    /* Determine the data offset. Factors include controller gen, and command */
    switch (cmd) {
    case CYPRESS_CONFIG_RETRIEVE_DATA_STRUCTURE:
        switch (cypress->silicon_gen) {
        case GEN_FOUR:
            data_offset = 3;
            break;
        case GEN_SIX:
            data_offset = 5;
            break;
        default:
            return -1;
        }
        break;
    case CYPRESS_CONFIG_GET_SELF_TEST_RESULTS:
        switch (cypress->silicon_gen) {
        case GEN_SIX:
            status_offset = 0;
            id_offset = 1;
            bytes_received_offset = 2;
            data_offset = 5;
            break;
        default:
            return -1;
        }
        break;
    default:
        mtouch_error(cypress->log_name, "Unsupported command %d", cmd);
        return -1;
    }

    do {
        buf[0] = HIGH_BYTE(total_bytes_received);
        buf[1] = LOW_BYTE(total_bytes_received);
        buf[2] = HIGH_BYTE(bytes_remaining);
        buf[3] = LOW_BYTE(bytes_remaining);
        buf[4] = data_id;

        if ((bytes_remaining + CONFIGURATION_COMMAND_DATA + data_offset) < CYPRESS_MAX_REGLEN)
            bytes_to_read = bytes_remaining + data_offset;
        else
            bytes_to_read = (CYPRESS_MAX_REGLEN - CONFIGURATION_COMMAND_DATA);

        if ((cypress_config_cmd(cypress, cmd, buf, sizeof(buf), &reply, bytes_to_read)) == -1) {
            mtouch_error (cypress->log_name, "Failed to get indexed data");
            goto fail;
        }

        /* NOTE: This check may not apply to other tests */
        if (-1 != status_offset) {
            switch (reply[status_offset]) {
            case 0x01:
                mtouch_error (cypress->log_name, "Command %d data ID[%d] failed", cmd, data_id);
                goto fail;
            case 0xFF:
                mtouch_error (cypress->log_name, "Unsupported self-test ID[%d]", data_id);
                goto fail;
            }
        }

        /* Ensure data returned matches requested data id */
        if (-1 != id_offset) {
            if (data_id != reply[id_offset]) {
                mtouch_error (cypress->log_name, "Data requested ID[%d] does not match data requested ID[%d]", reply[id_offset], data_id);
                goto fail;
            }
        }

        /* Add the reply to the data */
        bytes_received = ((reply[bytes_received_offset] << 8) | reply[bytes_received_offset + 1]);
        total_bytes_received += bytes_received;
        bytes_remaining -= bytes_received;

        /* Note: The last two bytes received in each transaction contains mutual-cap calibration data for panel, not sure what to do with that yet */

        /* Copy reply to results buffer */
        memcpy (data, &reply[data_offset], bytes_received);
        data += bytes_received;

        free (reply);
        reply = NULL;
    } while (bytes_remaining);

    return EOK;

fail:
    if (NULL != reply)
        free(reply);

    reply = NULL;

    return -1;
}

int
cypress_run_self_test(cypress_dev_t* cypress, uint8_t test_id, uint8_t reply_len, uint8_t *results, int results_len, int skip_setting_mode)
{
    uint8_t old_mode = 0;
    uint8_t *reply = NULL;
    int rc = EOK;

    /* Ensure we are in Config Test Mode */
    if (!skip_setting_mode) {
        if ((old_mode = cypress_get_mode(cypress)) != CONFIGURATION_AND_TEST_MODE) {
            rc = cypress_set_mode(cypress, CONFIGURATION_AND_TEST_MODE);
            if (-1 == rc) {
                mtouch_error (cypress->log_name, "Failed to set controller to config and test mode");
                return -1;
            }
        }
    }

    rc = cypress_config_cmd(cypress, CYPRESS_CONFIG_RUN_SELF_TEST, &test_id, CYPRESS_CONFIG_SELF_TEST_COMMAND_LEN, &reply, reply_len);
    if (-1 == rc) {
        mtouch_error (cypress->log_name, "Failed to run self test: %d", test_id);
        goto done;
    }

    /* Ensure self test passed (0x00 = Pass, 0x01 = Fail)*/
    if (reply[0]) {
        mtouch_error (cypress->log_name, "Self Test returned with a failure.");
        rc = -1;
        goto done;
    }

    /* Check to see if there is data pending, 0x00 = None, 0x01 = Data Pending */
    /* Note this is inconsistant with the docs.  Touchscreen Manufacturing Testing_001-85142_0E_V.PDF states 0x00 is OK */
    //if (!reply[1]) {
    if (reply[1]) {
        mtouch_warn (cypress->log_name, "Self Test ran, however no results are available");
        rc = -1;
        goto done;
    }

    /* Fetch the results */
    rc = cypress_get_indexed_data(cypress, CYPRESS_CONFIG_GET_SELF_TEST_RESULTS, test_id, results, results_len);
    if (-1 == rc) {
        mtouch_error (cypress->log_name, "Failed to get test results data from controller");
        goto done;
    }

    /* Restore Old Mode if applicable */
    if (!skip_setting_mode) {
        if (old_mode != CONFIGURATION_AND_TEST_MODE) {
            rc = cypress_set_mode(cypress, (old_mode));
            if (-1 == rc) {
                mtouch_error (cypress->log_name, "Failed to restore controller mode after config test: %s", __FUNCTION__);
                goto done;
            }
        }
    }

done:
    free (reply);
    reply = NULL;

    return rc;
}

int
cypress_read_cm_ref(cypress_dev_t* cypress)
{
    int i;

    if ((cypress->cm_panel_ref_max == NULL) && (cypress->cm_panel_ref_min == NULL)) {
        cypress->cm_panel_ref_min = (uint32_t **)malloc(sizeof(uint32_t *) * cypress->register_map.pcfg_data.electrodes_x);
        if (cypress->cm_panel_ref_min == NULL) {
            return -1;
        }

        cypress->cm_panel_ref_min[0] = (uint32_t *)malloc(sizeof(uint32_t) * cypress->register_map.pcfg_data.electrodes_x * cypress->register_map.pcfg_data.electrodes_y);
        if (cypress->cm_panel_ref_min[0] == NULL) {
            free(cypress->cm_panel_ref_min);
            cypress->cm_panel_ref_min = NULL;
            return -1;
        }

        for(i = 0; i < cypress->register_map.pcfg_data.electrodes_x; i++)
            cypress->cm_panel_ref_min[i] = (*(cypress->cm_panel_ref_min) + cypress->register_map.pcfg_data.electrodes_y * i);

        cypress->cm_panel_ref_max = (uint32_t **)malloc(sizeof(uint32_t *) * cypress->register_map.pcfg_data.electrodes_x);
        if (cypress->cm_panel_ref_max == NULL) {
            free(*(cypress->cm_panel_ref_min));
            free(cypress->cm_panel_ref_min);
            cypress->cm_panel_ref_min = NULL;
            return -1;
        }

        cypress->cm_panel_ref_max[0] = (uint32_t *)malloc(sizeof(uint32_t) * cypress->register_map.pcfg_data.electrodes_x * cypress->register_map.pcfg_data.electrodes_y);
        if (cypress->cm_panel_ref_max[0] == NULL) {
            free(cypress->cm_panel_ref_max);
            cypress->cm_panel_ref_max = NULL;

            free(*(cypress->cm_panel_ref_min));
            free(cypress->cm_panel_ref_min);
            cypress->cm_panel_ref_min = NULL;
            return -1;
        }

        for(i = 0; i < cypress->register_map.pcfg_data.electrodes_x; i++)
            cypress->cm_panel_ref_max[i] = (*(cypress->cm_panel_ref_max) + cypress->register_map.pcfg_data.electrodes_y * i);

        if (read_cm_ref(cypress->cm_panel_ref_min, cypress->cm_panel_ref_max,
            cypress->register_map.pcfg_data.electrodes_x, cypress->register_map.pcfg_data.electrodes_y) == -1) {
            free(*(cypress->cm_panel_ref_max));
            free(cypress->cm_panel_ref_max);
            cypress->cm_panel_ref_max = NULL;

            free(*(cypress->cm_panel_ref_min));
            free(cypress->cm_panel_ref_min);
            cypress->cm_panel_ref_min = NULL;
            return -1;
        }
    }

    return EOK;
}

int
cypress_run_cm_panel_test(cypress_dev_t* cypress, int skip_setting_mode, int save_to_file)
{
    uint16_t results_len = (2 * cypress->register_map.pcfg_data.electrodes_x * cypress->register_map.pcfg_data.electrodes_y + 2);
    uint32_t **Cm_matrix;
    uint8_t *results =(uint8_t *) calloc (results_len, sizeof (uint8_t));
    int offset = 0;
    int i, j;
    int cm_calib_required = 0;
    int count = 0;

    if (NULL == results) {
        mtouch_error (cypress->log_name, "Failed to allocate memory for Cm Panel self test results: %d", errno);
        return -1;
    }

    if (cypress->run_cm_panel_test) {
        if (cypress_read_cm_ref(cypress) == -1) {
            mtouch_error (cypress->log_name, "Failed to read reference data for Cm Panel");
            free(results);
            return -1;
        }
    }

    /* Only supported on Gen 6 Devices, specifically CYTAT81X Family */
    switch (cypress->silicon_gen) {
    case GEN_SIX:
        //if (GEN_SIX_ID_CYTAT81X == //ERICK TODO: Bootloader tells us this, however we are not keeping it
        break;
    default:
        mtouch_error(cypress->log_name, "Cm Panel Test only supported on Gen 6 devices, specifically CYTAT81X Family");
        cypress_update_pps_status(cypress, "cm_panel_test::unsupported");
        free(results);
        return -1;
    }

    if (-1 == (cypress_run_self_test (cypress, CYPRESS_CONFIG_CM_PANEL_TEST, CYPRESS_CONFIG_CM_PANEL_TEST_REPLY_LEN, results, results_len, skip_setting_mode))) {
        mtouch_error (cypress->log_name, "Cm Panel Test failed.  Note: Cm Panel Test is only supported on Gen 6 CYTAT81X Family of controllers.");
        cypress_update_pps_status(cypress, "cm_panel_test::unsupported");
        free(results);
        return -1;
    }

    /* Allocate the 2x2 Matrix */
    Cm_matrix  = (uint32_t **)malloc(sizeof(uint32_t *) * cypress->register_map.pcfg_data.electrodes_x);
    if (NULL == Cm_matrix) {
        mtouch_error (cypress->log_name, "Failed to allocate memory for the Cm_matrix");
        cypress_update_pps_status(cypress, "cm_panel_test::failed");
        return -1;
    }

    Cm_matrix[0] = (uint32_t *)malloc(sizeof(uint32_t) * cypress->register_map.pcfg_data.electrodes_y * cypress->register_map.pcfg_data.electrodes_x);
    if (NULL == Cm_matrix[0]) {
        mtouch_error (cypress->log_name, "Failed to allocate memory for the Cm_matrix");
        cypress_update_pps_status(cypress, "cm_panel_test::failed");
        return -1;
    }

    /* Sort out the arrays */
    for(i = 0; i < cypress->register_map.pcfg_data.electrodes_x; i++)
        Cm_matrix[i] = (*Cm_matrix + cypress->register_map.pcfg_data.electrodes_y * i);

    /* Calculate the Femto Farads */
    for (j = 0; j < cypress->register_map.pcfg_data.electrodes_y; j++) {
        for (i = 0; i < cypress->register_map.pcfg_data.electrodes_x; i++) {
            Cm_matrix[i][j] = ((results[offset + 1] << 8) | results[offset]) * 10;

            /* Check limits */
            if (cypress->run_cm_panel_test) {
                if ((cypress->detect_panel_detach) && (Cm_matrix[i][j] <= cypress->panel_detach_cm_value)) {
                    count++;
                }

                if ((Cm_matrix[i][j] > ((cypress->cm_panel_ref_max[i][j] * (100 + cypress->cm_panel_tolerance_max)) / 100))
                    || (Cm_matrix[i][j] < ((cypress->cm_panel_ref_min[i][j] * (100 - cypress->cm_panel_tolerance_min)) / 100))) {
                        if (cypress->run_calib_cm_fail)
                            cypress->calibration_required = 1;
                        cm_calib_required = 1;
                }
            }

            offset += 2;
        }
    }

    if ((cypress->panel_detach_detected) || ((cypress->detect_panel_detach) && (count >= cypress->panel_detach_cm_count))) {
        cypress->panel_detach_detected = 1;
        cypress->calibration_required = 0;
        cm_calib_required = 0;

        mtouch_info (cypress->log_name, "Panel Removal Detected");
    }

    if (save_to_file) {
        if (save_cm_file(CM_FILE, cypress, Cm_matrix) == -1) {
            mtouch_error (cypress->log_name, "Failed to save data for Cm Panel");
        }
    }

    /* Free results */
    free (results);

    /* Free Cm_matrix */
    free (*Cm_matrix);
    free (Cm_matrix);


    if (cypress->run_cm_panel_test) {
		if(cm_calib_required)
           cypress_update_pps_status(cypress, "cm_panel_test::calibration_required");
	    else
		   cypress_update_pps_status(cypress, "cm_panel_test::no_calibration_required");
         //cypress_update_pps_status(cypress,(const char *) cm_calib_required ? "cm_panel_test::calibration_required" : "cm_panel_test::no_calibration_required");


        if (cypress->verbose)
            mtouch_info (cypress->log_name, "%s", cm_calib_required ? "cm_panel_test::calibration_required" : "cm_panel_test::no_calibration_required");

    }

	return EOK;
}

int
cypress_read_cp_ref(cypress_dev_t* cypress)
{
    if ((cypress->cp_panel_tx_ref_max == NULL) && (cypress->cp_panel_tx_ref_min == NULL)) {
        cypress->cp_panel_tx_ref_max = (uint32_t *)malloc(sizeof(uint32_t) * cypress->register_map.pcfg_data.electrodes_y);
        if (cypress->cp_panel_tx_ref_max == NULL) {
            return -1;
        }

        cypress->cp_panel_tx_ref_min = (uint32_t *)malloc(sizeof(uint32_t) * cypress->register_map.pcfg_data.electrodes_y);
        if (cypress->cp_panel_tx_ref_min == NULL) {
            free(cypress->cp_panel_tx_ref_max);
            cypress->cp_panel_tx_ref_max = NULL;
            return -1;
        }

        if (read_cp_tx_ref(cypress->cp_panel_tx_ref_min, cypress->cp_panel_tx_ref_max, cypress->register_map.pcfg_data.electrodes_y) == -1) {
            free(cypress->cp_panel_tx_ref_max);
            cypress->cp_panel_tx_ref_max = NULL;
            free(cypress->cp_panel_tx_ref_min);
            cypress->cp_panel_tx_ref_min = NULL;
            return -1;
        }
    }

    if ((cypress->cp_panel_rx_ref_max == NULL) && (cypress->cp_panel_rx_ref_min == NULL)) {
        cypress->cp_panel_rx_ref_max = (uint32_t *)malloc(sizeof(uint32_t) * cypress->register_map.pcfg_data.electrodes_x);
        if (cypress->cp_panel_rx_ref_max == NULL) {
            return -1;
        }

        cypress->cp_panel_rx_ref_min = (uint32_t *)malloc(sizeof(uint32_t) * cypress->register_map.pcfg_data.electrodes_x);
        if (cypress->cp_panel_rx_ref_min == NULL) {
            free(cypress->cp_panel_rx_ref_max);
            cypress->cp_panel_rx_ref_max = NULL;
            return -1;
        }

        if (read_cp_rx_ref(cypress->cp_panel_rx_ref_min, cypress->cp_panel_rx_ref_max, cypress->register_map.pcfg_data.electrodes_x) == -1) {
            free(cypress->cp_panel_rx_ref_max);
            cypress->cp_panel_rx_ref_max = NULL;
            free(cypress->cp_panel_rx_ref_min);
            cypress->cp_panel_rx_ref_min = NULL;
            return -1;
        }
    }

    return EOK;
}

int
cypress_run_cp_panel_test(cypress_dev_t* cypress, int skip_setting_mode, int save_to_file)
{
    uint16_t results_len = ((cypress->register_map.pcfg_data.electrodes_x * 4) + (cypress->register_map.pcfg_data.electrodes_y * 4)); //See below...

    /* Big old note here:

    The results len is according to the doc "Touchscreen Manufacturing Testing_001-85142_0E_V.PDF" Section 3.2.12.4 Procedure for CYAT81X Family
    is obtained by the following equation:

             Data size = (TX + RX + TX_CAL + RX_CAL)*2

    TX and RX are obtained from the System information mode: ELECTRODES_X (which is TX) and ELECTRODES_Y (which is RX).

    However obtaining the information on how to properly obtain TX_CAL and RX_CAL has been very difficult.

    An answer I got from one of the Cypress folks was:

    [clip]
    First of all, Sensor configuration is 20TX * 26RX for this project.
    For the order of reading Cp value by the command along with the guidance from the document(Testing_001-85142_0E_V.PDF),
    TX0 -> TX0cal -> TX1 -> TX1cal ,,,,, ->TX19 -> TX19cal -> RX0 -> RX0cal -> RX1 ->RX1cal,,,, -> RX25 -> RX25cal.
    Herein, you don’t have to compare TX_CAL/RX_CAL data since the criteria to be compared with panel Cp value has only TX/RX Cp value excluding CAL data.
    Thus, you just need parsing sensor(TX/RX) Cp data and compare with reference(criteria) value.

    The command to read Cp values is same as noted in the doc.
    w 24 00 20 00 08 00 00 00 B8 06 p
    r 24 x x x ,,,, (0xB8 = 184 bytes)
    [end clip]

    The example given used TX = TX_CAL and RX = RX_CAL

        Data size = 20(TX) + 26(RX) + 20(TX_CAL) + 26(RX_CAL)

    This does however tie in with the last note in Section 3.2.12.4:

    [clip]
    ; Return Data – A two-byte self-capacitance value for each TX (size = 2*NUM_TX), followed ;by
    a two-byte self-capacitance calibration value for each TX (size = 2*NUM_TX). A two-;byte self-
    capacitance value for each RX (size = 2*NUM_RX), followed by a two-byte self-;capacitance
    calibration value for each RX (size = 2*NUM_RX).The self-capacitance value ;must be multiplied
    by decimal 10 to get femto-farads.
    [end clip]

    So given this we should then in theory receieve: (TX * 4) + (RX * 4)

    ¯\_(ツ)_/¯

    */

    uint32_t *Cp_tx_selfcap =(uint32_t *) calloc(cypress->register_map.pcfg_data.electrodes_y, sizeof(uint32_t));
    uint32_t *Cp_tx_selfcap_cal =(uint32_t *) calloc(cypress->register_map.pcfg_data.electrodes_y, sizeof(uint32_t));
    uint32_t *Cp_rx_selfcap =(uint32_t *) calloc(cypress->register_map.pcfg_data.electrodes_x, sizeof(uint32_t));
    uint32_t *Cp_rx_selfcap_cal =(uint32_t *) calloc(cypress->register_map.pcfg_data.electrodes_x, sizeof(uint32_t));

    /* Note: Cp_*x_selfcap results are in femto-farads */

    uint8_t *results =(uint8_t *) calloc (results_len, sizeof (uint8_t));
    int offset = 0;
    int i;
    int cp_calib_required = 0;
    int count = 0;
    int rc = 0;

    if ((NULL == Cp_tx_selfcap_cal) || (NULL == Cp_rx_selfcap_cal) || (NULL == Cp_tx_selfcap) || (NULL == Cp_rx_selfcap)) {
        mtouch_error (cypress->log_name, "Failed to allocate memory for Cp Panel self test: %d", errno);
        rc = -1;
        goto done;
    }

    if (cypress->run_cp_panel_test) {
        if (cypress_read_cp_ref(cypress) == -1) {
            mtouch_error (cypress->log_name, "Failed to read reference data for Cp Panel");
            rc = -1;
            goto done;
        }
    }

    /* Only supported on Gen 6 Devices, specifically CYTAT81X Family */
    switch (cypress->silicon_gen) {
    case GEN_SIX:
        //if (GEN_SIX_ID_CYTAT81X == //ERICK TODO: Bootloader tells us this, however we are not keeping it
        break;
    default:
        mtouch_error(cypress->log_name, "Cp Panel Test only supported on Gen 6 devices, specifically CYTAT81X Family");
        cypress_update_pps_status(cypress, "cp_panel_test::unsupported");
        rc = -1;
        goto done;
    }

    if (-1 == (cypress_run_self_test (cypress, CYPRESS_CONFIG_CP_PANEL_TEST, CYPRESS_CONFIG_CP_PANEL_TEST_REPLY_LEN, results, results_len, skip_setting_mode))) {
        mtouch_error (cypress->log_name, "Cp Panel Test failed.  Note: Cp Panel Test is only supported on Gen 6 CYTAT81X Family of controllers.");
        cypress_update_pps_status(cypress, "cp_panel_test::unsupported");
        rc = -1;
        goto done;
    }

    for (i = 0; i < cypress->register_map.pcfg_data.electrodes_y; i++) {
        /* Calculate the Femto Farads and compare to reference value, see docs for algorithm */
        Cp_tx_selfcap[i] = ((results[offset + 1] << 8) | results[offset]) * 10;
        offset += 2;

        if (cypress->run_cp_panel_test) {
            if ((Cp_tx_selfcap[i] > ((cypress->cp_panel_tx_ref_max[i] * (100 + cypress->cp_panel_tolerance_max)) / 100))
                || (Cp_tx_selfcap[i] < ((cypress->cp_panel_tx_ref_min[i] * (100 - cypress->cp_panel_tolerance_min)) / 100))) {
                if (cypress->run_calib_cp_fail)
                    cypress->calibration_required = 1;
                cp_calib_required = 1;
            }
        }
    }

    for (i = 0; i < cypress->register_map.pcfg_data.electrodes_y; i++) {
        Cp_tx_selfcap_cal[i] = ((results[offset + 1] << 8) | results[offset]);
        offset += 2;
    }

    for (i = 0; i < cypress->register_map.pcfg_data.electrodes_x; i++) {
        /* Calculate the Femto Farads and compare to reference value, see docs for algorithm */
        Cp_rx_selfcap[i] = ((results[offset + 1] << 8) | results[offset]) * 10;
        offset += 2;

        if (cypress->run_cp_panel_test) {
            /* Check to ensure that the there are not too many lines are out of spec, if so the panel may be detached and we don't want to perform a calibration */
            if ((cypress->detect_panel_detach) && (Cp_rx_selfcap[i] <= cypress->panel_detach_cp_value)) {
                count++;
            }

            if ((Cp_rx_selfcap[i] > ((cypress->cp_panel_rx_ref_max[i] * (100 + cypress->cp_panel_tolerance_max)) / 100))
                || (Cp_rx_selfcap[i] < ((cypress->cp_panel_rx_ref_min[i] * (100 - cypress->cp_panel_tolerance_min)) / 100))) {
                if (cypress->run_calib_cp_fail)
                    cypress->calibration_required = 1;
                cp_calib_required = 1;
            }
        }
    }

    for (i = 0; i < cypress->register_map.pcfg_data.electrodes_x; i++) {
        Cp_rx_selfcap_cal[i] = ((results[offset + 1] << 8) | results[offset]);
        offset += 2;
    }

    if ((cypress->panel_detach_detected) || ((cypress->detect_panel_detach) && (count >= cypress->panel_detach_cp_count))) {
        cypress->panel_detach_detected = 1;
        cypress->calibration_required = 0;
        cp_calib_required = 0;

        mtouch_info (cypress->log_name, "Panel Removal Detected");
    }

    if (save_to_file) {
        if (save_cp_file(CP_FILE, cypress, Cp_tx_selfcap, Cp_tx_selfcap_cal, Cp_rx_selfcap, Cp_rx_selfcap_cal) == -1) {
            mtouch_error (cypress->log_name, "Failed to save data for Cp Panel");
        }
    }

    if (cypress->run_cp_panel_test) {

		if(cp_calib_required)
           cypress_update_pps_status(cypress, "cp_panel_test::calibration_required");
	    else
		   cypress_update_pps_status(cypress, "cp_panel_test::no_calibration_required");
           //cypress_update_pps_status(cypress, cp_calib_required ? "cp_panel_test::calibration_required" : "cp_panel_test::no_calibration_required");

        if (cypress->verbose)
            mtouch_info (cypress->log_name, "%s", cp_calib_required ? "cp_panel_test::calibration_required" : "cp_panel_test::no_calibration_required");
    }

done:
    /* Free results */
    free (results);

    /* Free Cp_matrix */
    free (Cp_tx_selfcap);
    free (Cp_tx_selfcap_cal);
    free (Cp_rx_selfcap);
    free (Cp_rx_selfcap_cal);

    return rc;
}

int16_t
cypress_retrieve_idac_read_length(cypress_dev_t* cypress)
{
    int16_t read_length = 0;

    if (0 == cypress->sensing_mode) {
        mtouch_error (cypress->log_name, "Invalid sensing mode specified.  Unable to fetch IDAC read length.");
        return -1;
    }

    if (cypress->sensing_mode & CTRL_SENSING_MODE_MUTUAL_CAP_FINE) {
        read_length = (cypress->register_map.pcfg_data.electrodes_x * cypress->register_map.pcfg_data.electrodes_y + 1);
    }
    if (cypress->sensing_mode & CTRL_SENSING_MODE_SELF_CAP) {
        read_length = (cypress->register_map.pcfg_data.electrodes_x * cypress->register_map.pcfg_data.electrodes_y + 2);
    }
    if (cypress->sensing_mode & CTRL_SENSING_MODE_MUTUAL_CAP_BUTTON) {
        read_length = (2 * (cypress->register_map.operating_mode_info->num_btns + 1));
    }

    return read_length;
}

int
cypress_retrieve_idac_data(cypress_dev_t* cypress, uint8_t **reply)
{
    int16_t read_length = 0;
    uint16_t read_offset = 0;
    uint8_t data_id = 0;
    uint8_t buf[5] = {0};
    uint8_t old_mode = 0;
    int rc = 0;

    //read_length = cypress_retrieve_idac_read_length(cypress);
    if (-1 == (read_length = cypress_retrieve_idac_read_length(cypress))) {
        return -1;
    }

    /* Ensure we are in Config Test Mode */
    if ((old_mode = cypress_get_mode(cypress)) != CONFIGURATION_AND_TEST_MODE) {
        rc = cypress_set_mode(cypress, CONFIGURATION_AND_TEST_MODE);
        if (-1 == rc) {
            mtouch_error (cypress->log_name, "Failed to set controller to config and test mode");
            return -1;
        }
    }

    /* Note: Only supports Center Frequency at this time for Gen 6 controllers */

    /* Data ID and Read Offsets */
    switch (cypress->silicon_gen) {
    case GEN_FOUR:
        if (cypress->sensing_mode & CTRL_SENSING_MODE_MUTUAL_CAP_FINE) {
            read_offset = 0;
            data_id = 0x00;
        } else if (cypress->sensing_mode & CTRL_SENSING_MODE_SELF_CAP) {
            data_id = 0x01;
            /* This has two passes as there are TX and RX data */
            /* Pass one RX */
            read_offset = 0;

            /* Pass two TX */
            read_offset = 1;
        } else if (cypress->sensing_mode & CTRL_SENSING_MODE_MUTUAL_CAP_BUTTON) {
            read_offset = 0;
            data_id = 0x03;
        }
        break;
    case GEN_SIX:
        if (cypress->sensing_mode & CTRL_SENSING_MODE_MUTUAL_CAP_FINE) {
            read_offset = 1;
        } else if (cypress->sensing_mode & CTRL_SENSING_MODE_SELF_CAP) {
            /* This has two passes as there are TX and RX data */
            /* Pass one RX */
            //read_offset = 3;

            /* Pass two TX */
            read_offset = 5;
        } else if (cypress->sensing_mode & CTRL_SENSING_MODE_MUTUAL_CAP_BUTTON) {
            read_offset = 7;

            /* TODO ERICK: Docs have IDAC Button Self as a retrievable structure yet we cannot request calibration for it */
        }

        data_id = 0x00;
        break;
    default:
        mtouch_error (cypress->log_name, "Unrecognized silicon generation %d", cypress->silicon_gen);
        return -1;
    }

    buf[0] = ((read_offset & 0xFF00) >> 8);
    buf[1] = (read_offset & 0x00FF);
    buf[2] = ((read_length & 0xFF00) >> 8);
    buf[3] = (read_length & 0x00FF);
    buf[4] = data_id;

    if ((cypress_config_cmd(cypress, CYPRESS_CONFIG_RETRIEVE_DATA_STRUCTURE, buf, sizeof(buf), reply, 0)) == -1) {
        mtouch_error (cypress->log_name, "Failed to read IDAC data");
        return -1;
    }

    /* Restore Old Mode if applicable */
    if (old_mode != CONFIGURATION_AND_TEST_MODE) {
        rc = cypress_set_mode(cypress, (old_mode));
        if (-1 == rc) 
        {
            mtouch_error (cypress->log_name, "Failed to restore the Old Mode");
            return -1;
        }
    }

    /* Get the actual read length from the returned packet */

    return read_length;
}

int
cypress_init_baselines(cypress_dev_t* cypress)
{
    uint8_t buf[3] = {0, 0, 0};
    uint8_t *reply = NULL;

    /* Initalize Baselines */
    buf[0] = (cypress->sensing_mode & SENSING_MASK);
    if ((cypress_config_cmd(cypress, CYPRESS_CONFIG_INITALIZE_BASELINES, buf, sizeof(buf), &reply,
                            CYPRESS_CONFIG_INITALIZE_BASELINES_REPLY_LEN)) == -1) {
        mtouch_error (cypress->log_name, "Baseline Initialization failed");
        return -1;
    }

    mtouch_info (cypress->log_name, "Baseline Initialization successful");

    return EOK;
}

int
cypress_calibrate_idacs(cypress_dev_t* cypress)
{
    uint8_t buf[3] = {0, 0, 0};
    uint8_t *reply = NULL;
    uint8_t old_mode = 0;
    int rc = 0;

    cypress->idac_calibration_status = -1; /* -1 = Fail, 0 = Not Run, 1 = Success */

    if (cypress->sensing_mode) {
        /* Ensure we are in Config Test Mode */
        if ((old_mode = cypress_get_mode(cypress)) != CONFIGURATION_AND_TEST_MODE) {
            rc = cypress_set_mode(cypress, CONFIGURATION_AND_TEST_MODE);
            if (-1 == rc) {
                mtouch_error (cypress->log_name, "Failed to set controller to config and test mode");
                return -1;
            }
        }

        /* Calibrate IDACs for each mode that we are told is used */
        if (cypress->sensing_mode & CTRL_SENSING_MODE_MUTUAL_CAP_FINE) {
            buf[0] = SENSING_MODE_MUTUAL_CAP_FINE;

            if ((cypress_config_cmd(cypress, CYPRESS_CONFIG_CALIBRATE_IDACS, buf, sizeof(buf), &reply,
                                    CYPRESS_CONFIG_CALIBRATE_IDACS_REPLY_LEN)) == -1) {
                mtouch_error (cypress->log_name, "IDAC Calibration failed");
                return -1;
            }
        }
        if (cypress->sensing_mode & CTRL_SENSING_MODE_MUTUAL_CAP_BUTTON) {
            buf[0] = SENSING_MODE_MUTUAL_CAP_BUTTON;

            if ((cypress_config_cmd(cypress, CYPRESS_CONFIG_CALIBRATE_IDACS, buf, sizeof(buf), &reply,
                                    CYPRESS_CONFIG_CALIBRATE_IDACS_REPLY_LEN)) == -1) {
                mtouch_error (cypress->log_name, "IDAC Calibration failed");
                return -1;
            }
        }
        if (cypress->sensing_mode & CTRL_SENSING_MODE_SELF_CAP) {
            buf[0] = SENSING_MODE_SELF_CAP;

            if ((cypress_config_cmd(cypress, CYPRESS_CONFIG_CALIBRATE_IDACS, buf, sizeof(buf), &reply,
                                    CYPRESS_CONFIG_CALIBRATE_IDACS_REPLY_LEN)) == -1) {
                mtouch_error (cypress->log_name, "IDAC Calibration failed");
                return -1;
            }
        }

        mtouch_info (cypress->log_name, "IDAC Calibration successful");

        /* Init Baselines */
        if (EOK != cypress_init_baselines(cypress)) {
            return -1;
        }

        cypress->calibration_required = 0;
        cypress->idac_calibration_status = 1;

        /* Restore Old Mode if applicable */
        if (old_mode != CONFIGURATION_AND_TEST_MODE) {
            rc = cypress_set_mode(cypress, (old_mode));
            if (-1 == rc)
            {
                mtouch_error (cypress->log_name, "Failed to restore the Old Mode");
                return -1;
            }
        }
    }

    return EOK;
}

int
cypress_config_reply_ready(cypress_dev_t* cypress)
{
    uint8_t data;
    int i;

    for (i = 0; i < cypress->configuration_reply_retry_limit; i++) {
        if ((cypress->i2c_funcs.read_reg(cypress->i2c_fd, CONFIGURATION_COMMAND, sizeof(uint8_t), &data)) < 0) {
            mtouch_error (cypress->log_name, "Failed to read touch controller status");
            goto fail;
        }

        if (data & COMMAND_COMPLETE) {
            if (cypress->verbose > 5) {
                mtouch_info (cypress->log_name, "Controller has processed command, response awaiting");

                if (cypress->verbose > 6)
                    mtouch_info (cypress->log_name, "Reply delay for configuration command (msec): %d, Retry count: %d",
                                cypress->configuration_reply_retry_delay, i);
            }

            return EOK;
        }

        delay (cypress->configuration_reply_retry_delay);
    }

    mtouch_warn (cypress->log_name, "Controller took to long too reply to Configuration command");

fail:

    return -1;
}

int
cypress_read_configuration_reply(cypress_dev_t* cypress, uint8_t cmd, uint8_t **reply, uint16_t len)
{
    uint8_t *buf;
    int data_offset = 0;
    int bytes_to_read = 0;
    int bytes_remaining = len;

    if (*reply != NULL) {
        free (*reply);
        *reply = NULL;
    }

    *reply =(uint8_t *) calloc (len, sizeof (uint8_t));

    if (*reply == NULL) {
        mtouch_error (cypress->log_name, "Failed to allocate memory for configuration data response");
        return -1;
    }

    do {
        bytes_to_read = 0;
        /* Reply might be larger than the max read */
        if ((bytes_remaining + CONFIGURATION_COMMAND_DATA) > CYPRESS_MAX_REGLEN) {
            bytes_to_read = (CYPRESS_MAX_REGLEN - CONFIGURATION_COMMAND_DATA);
        } else {
            bytes_to_read += bytes_remaining;
        }

        /* Read the reply from the command */
        buf =(uint8_t *) calloc(bytes_to_read + CONFIGURATION_COMMAND_DATA, sizeof(uint8_t));
        if ((cypress->i2c_funcs.read_reg(cypress->i2c_fd, HST_MODE, (bytes_to_read + CONFIGURATION_COMMAND_DATA), buf)) < 0) {
            mtouch_error (cypress->log_name, "Failed to read Retrieve Data output");
            return -1;
        }

        if (cypress->verbose > 5) {
            for(int i=0; i< (bytes_to_read + CONFIGURATION_COMMAND_DATA); i++)
                mtouch_error (cypress->log_name, "buf[%d]: %d", i, buf[i]);
        }

        memcpy ((*reply + data_offset), (buf + CONFIGURATION_COMMAND_DATA), bytes_to_read);
        bytes_remaining -= bytes_to_read;
        data_offset += bytes_to_read;

        if (cypress->verbose > 5)
            cypress_display_packet(cypress, "r", HST_MODE, buf, (bytes_to_read + CONFIGURATION_COMMAND_DATA));

    } while (bytes_remaining > 0);

    free(buf);
    return len;
}

void cypress_dump_haptic_data(cypress_dev_t* cypress, uint8_t *data, uint8_t len)
{
    char *buf;
    int i;

    buf = (char *) calloc (1, 3*len);
    if (buf == NULL) {
        mtouch_error (cypress->log_name, "Failed to allocate buffer space to dump haptic data");
        return;
    }

    for (i = 0; i < len; i++) {
        sprintf ((buf + strlen(buf)), "%x ", data[i]);
    }

    mtouch_debug(cypress->log_name,"haptic data = %s", buf);

    free(buf);
}

int
cypress_config_cmd(cypress_dev_t* cypress, uint8_t cmd, uint8_t *data, uint8_t data_len, uint8_t **reply, uint16_t reply_len)
{
    int rc = 0;
    uint8_t *buf;
    uint8_t reg;

    if ((cypress->i2c_funcs.read_reg(cypress->i2c_fd, HST_MODE, sizeof(uint8_t), &reg)) < 0) {
        mtouch_error (cypress->log_name, "Failed to read touch controller HST_MODE");
        return -1;
    }

    /* Write out the command.  Configuration and Test register does not appear to work correctly if you
       attempt to write to an offset into the register so you need to write everything back to it */
    data_len += 3;
    buf =(uint8_t *) calloc (data_len, sizeof (uint8_t));

    if (NULL == buf) {
        mtouch_error (cypress->log_name, "Failed to allocate memory for Configuration & Test command");
        return -1;
    }

    /* Ensure Command_complete is unset */
    cmd &= ~COMMAND_COMPLETE;
    buf[HST_MODE] = reg;    /* Mode */
    buf[1] = 0x00;  /* Reset */
    buf[2] = cmd;

    if (NULL != data)
        memcpy (&(buf[3]), data, (data_len - 3));

   if (cypress->verbose > 5)
        cypress_dump_haptic_data(cypress, buf, data_len);

    rc = cypress->i2c_funcs.write_reg(cypress->i2c_fd, HST_MODE, data_len, buf);
    if (0 != rc) {
        mtouch_error (cypress->log_name, "Failed to write command parameter");
        rc = -1;
        goto done;
    }

    if ((rc = cypress_config_reply_ready(cypress))) {
        if (-1 == rc) {
            mtouch_error (cypress->log_name, "Config_Test command %x failed", cmd);
            goto done;
        }
    }

    /* Fetch the reply from the controller */
    if ((rc = cypress_read_configuration_reply(cypress, cmd, reply, reply_len))) {
        if (-1 == rc) {
            mtouch_error (cypress->log_name, "Failed to read configuration reply");
            goto done;
        }
    }

done:
    free (buf);

    return rc;
}

int
cypress_read_config_param_flash(cypress_dev_t* cypress, uint8_t addr, uint8_t *data, uint8_t len)
{
    uint8_t old_mode = 0;
    int rc = 0;
    uint8_t *reply = NULL;

    /* Ensure we are in Config Test Mode */
    if ((old_mode = cypress_get_mode(cypress)) != CONFIGURATION_AND_TEST_MODE) {
        rc = cypress_set_mode(cypress, CONFIGURATION_AND_TEST_MODE);
        if (-1 == rc) {
            mtouch_error (cypress->log_name, "Failed to set controller to config and test mode");
            return -1;
        }
    }

    /* Get Config Block size (if unknown) */
    if (!cypress->config_block_size) {
        if ((cypress_config_cmd(cypress, CYPRESS_CONFIG_GET_ROW_SIZE, NULL, 0, &reply, CYPRESS_CONFIG_GET_ROW_SIZE_LEN)) == -1) {
            mtouch_error (cypress->log_name, "Failed to get row size");
            goto fail;
        }

        cypress->config_block_size = reply[0];
    }

    /* Restore Old Mode if applicable */
    if (old_mode != CONFIGURATION_AND_TEST_MODE) {
        rc = cypress_set_mode(cypress, (old_mode));
        if (-1 == rc) 
        {
            mtouch_error (cypress->log_name, "Failed to restore the Old Mode");
            return -1;
        }
    }

    return 0;

fail:
    if (NULL != reply)
        free (reply);

    return -1;
}

int
cypress_write_config_param_flash(cypress_dev_t* cypress, uint8_t addr, uint8_t *data, uint8_t len)
{
    return EOK;
}

#if defined(__QNXNTO__) && defined(__USESRCVERSION)
#include <sys/srcversion.h>
__SRCVERSION("$URL: http://svn.ott.qnx.com/product/branches/7.0.0/trunk/hardware/mtouch/cypress/configuration_test.c $ $Rev: 888004 $")
#endif
